package de.hwrBerlin.tools.logger;

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.PrintWriter;
import java.text.SimpleDateFormat;
import java.util.Date;

import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.MessageBox;
import org.eclipse.swt.widgets.Shell;

/**
 * Diese Klasse stellt einen universellen Logger bereit, der auf, durch ein Enum definierte, LogLevel beruht. Zustzlich soll dieser Logger Wert auf
 * eine benutzerfreundliche Oberflche legen.
 * 
 * @author Benni
 * 
 */

public abstract class Logger {
    
    private static String _logFilePath = "";
    private static String _logFileName = "dataMiningLog.txt";
    private static LogLevel _currentLogLevel = LogLevel.INFO;
    
    // Schalter der bestimmt, ob auch eine Fehlermeldung erscheinen soll.
    private static boolean graphicalLog = true;
    
    
    
    /**
     * Einfachste Logging-Methode. Loggt, wenn das LogLevel stimmt in die normale Konsole, in eine Datei und wenn LogLevel Error ist auch als ein
     * GUI-Fenster
     * 
     * @param pLogLevel
     * @param pLogMessage
     */
    public static void log(LogLevel pLogLevel, String pLogMessage) {
        // Setze LogMessage zusammen
        String logMessageToPrint = pLogLevel.toString() + ": " + pLogMessage;
        
        // Logge nur wenn das angegbene Level hher oder gleich dem gesetzten Level ist.
        if (pLogLevel.ordinal() >= _currentLogLevel.ordinal()) {
            System.out.println(logMessageToPrint);
            logToFile(logMessageToPrint);
        }
        
        // Sollte ein Error Level vorliegen zeige auch dem Nutzer eine grafische Meldung
        if (pLogLevel.equals(LogLevel.ERROR)) {
            showSWTErrorMessage(logMessageToPrint, "Fehler in der Anwendung");
        }
    }
    
    
    
    /**
     * Logging Methode mit zustzlichen Optionen, wie Objekt bergabe und Exception. Auerdem kann ein header fr die Grafische Fehlermeldung
     * mitgegeben werden. Sollten die zustzlichen Parameter null sein werden diese ignoriert.
     * 
     * @param pLogLevel
     * @param pLogMessage
     */
    public static void log(LogLevel pLogLevel, String pLogMessage, String pGuiTitle, String pClassName, Exception pExceptionParameter) {
        // Setze LogMessage zusammen
        StringBuffer logMessageToPrint = new StringBuffer();
        logMessageToPrint.append(pLogLevel.toString() + ": " + pLogMessage);
        
        // Wurde ein Klassenname angegeben?
        if (!stringIsNullOrEmpty(pClassName)) {
            logMessageToPrint.append("\nIn der Klasse: \n" + pClassName);
        }
        
        // Wurde eine Exception mit eingetragen?
        if (pExceptionParameter != null) {
            logMessageToPrint.append("\nEs wurde folgende Exception ausgelst: \n" + pExceptionParameter);
            // Printe den StackTrace
            // TODO: Ich habe noch nicht herausgefunden wie man den gesamten StackTrace printen kann
            for (int i = 0; i < pExceptionParameter.getStackTrace().length; i++) {
                logMessageToPrint.append("\n" + pExceptionParameter.getStackTrace()[i].toString());
            }
        }
        
        // Logge nur wenn das angegbene Level hher oder gleich dem gesetzten Level ist.
        if (pLogLevel.ordinal() >= _currentLogLevel.ordinal()) {
            System.out.println(logMessageToPrint.toString());
            logToFile(logMessageToPrint.toString());
        }
        
        // Sollte ein Error Level vorliegen zeige auch dem Nutzer eine grafische Meldung
        if (pLogLevel.equals(LogLevel.ERROR)) {
            String guiHeader = "Fehler in der Anwendung";
            // Soll ein anderer Header verwendet werden?
            if (!stringIsNullOrEmpty(pGuiTitle)) {
                guiHeader = pGuiTitle;
            }
            showSWTErrorMessage(logMessageToPrint.toString(), guiHeader);
        }
    }
    
    
    
    /**
     * Hilfsmethode um in eine Datei zu loggen. Die LogMessages werden immer zeilenweise angehngt. Die Uhrzeit wird automatisch mit hinzugezogen.
     * Sollte in der logMessage ein Zeilenumbruch gefordert sein "\n" dann wird eine neue Zeile geprintet.
     * 
     * @param pLogMessage
     */
    private static void logToFile(String pLogMessage) {
        File file = new File(_logFilePath + _logFileName);
        PrintWriter pw;
        
        // Wenn die Datei nicht existiert dann erstelle sie
        if (!file.exists()) {
            try {
                // PrintWriter ohne Append (false)
                pw = new PrintWriter(new FileWriter(file, false), true);
                pw.close();
            }
            catch (IOException e) {
                System.out.println("ERROR: Konnte kein LogFile erstellen! Pfad: " + _logFilePath + _logFileName);
            }
        }
        
        // Schreibe ans Ende der Datei
        try {
            // PrintWriter mit Append (true)
            pw = new PrintWriter(new FileWriter(file, true), true);
            // Datum finden und richtig formatieren
            SimpleDateFormat dateFormat = new SimpleDateFormat("dd.MM.yyyy HH:mm:ss");
            Date nowDate = new Date(System.currentTimeMillis());
            // Sind Zeilenumbrche gefordert?
            if (pLogMessage.contains("\n")) {
                pw.println(dateFormat.format(nowDate) + ": " + pLogMessage.split("\n")[0]);
                for (int i = 1; i < pLogMessage.split("\n").length; i++) {
                    pw.println(pLogMessage.split("\n")[i]);
                }
            }
            else {
                pw.println(dateFormat.format(nowDate) + ": " + pLogMessage);
            }
            // Zum schluss eine leere Zeile zur bersicht
            pw.println();
            pw.close();
        }
        
        catch (IOException e) {
            System.out.println("ERROR: Konnte nichts loggen!");
        }
    }
    
    
    
    /**
     * Erzeugt ein GUI-Fenster in dem eine Fehlermeldung steht.
     * 
     * @param pLogMessage
     *            Die Nachricht im Rumpf
     * @param pFrameHeader
     *            Der Titel des Fehlerfensters
     */
    private static void showSWTErrorMessage(String pLogMessage, String pFrameHeader) {
        // Wurde das graphische Logging eingestellt?
        if (graphicalLog) {
            // Zeige Fehlermeldung nur wenn nicht schon ein Fenster offen ist
            if (!isNull(Display.getCurrent())) {
                Display display = new Display();
                Shell shell = new Shell(display);
                MessageBox msgBox = new MessageBox(shell, SWT.ICON_ERROR);
                msgBox.setMessage(pLogMessage);
                msgBox.setText(pFrameHeader);
                msgBox.open();
            }
            // Es ist ein Fenster offen, reihe Nachricht ein.
            else {
                MessageBox msgBox = new MessageBox(Display.getCurrent().getActiveShell(), SWT.ICON_ERROR);
                msgBox.setMessage(pLogMessage);
                msgBox.setText(pFrameHeader);
                msgBox.open();
            }
        }
    }
    
    
    
    public static String getLogFilePath() {
        return _logFilePath;
    }
    
    
    
    public static void setLogFilePath(String logFilePath) {
        Logger._logFilePath = logFilePath;
    }
    
    
    
    public static String getLogFileName() {
        return _logFileName;
    }
    
    
    
    public static void setLogFileName(String logFileName) {
        Logger._logFileName = logFileName;
    }
    
    
    
    public static LogLevel getCurrentLogLevel() {
        return _currentLogLevel;
    }
    
    
    
    public static void setCurrentLogLevel(LogLevel currentLogLevel) {
        Logger._currentLogLevel = currentLogLevel;
    }
    
    
    
    // Hilfsmethode um zu prfen ob ein Object null ist, sehr unsauber aber funktional
    private static boolean isNull(Object pObject) {
        try {
            pObject.getClass();
            return true;
        }
        catch (Exception e) {
            return false;
        }
    }
    
    
    
    public static boolean isGraphicalLog() {
        return graphicalLog;
    }
    
    
    
    public static void setGraphicalLog(boolean graphicalLog) {
        Logger.graphicalLog = graphicalLog;
    }
    
    
    
    // Hilfsmethode um zu prfen ob ein String leer oder null ist
    private static boolean stringIsNullOrEmpty(String pTestString) {
        if (pTestString == null || pTestString.trim().equals("")) {
            return true;
        }
        else {
            return false;
        }
    }
}
