package de.hwrBerlin.tools.configManager;


import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.PrintWriter;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.Map;

import org.omg.CORBA._PolicyStub;

import de.hwrBerlin.tools.logger.LogLevel;
import de.hwrBerlin.tools.logger.Logger;

/**
* Diese Klasse kann Konfigurationsdateien lesen. Ausgegeben werden die Konfigurationen in Form einer Map.
* Kommentarzeilen werden ignoriert (Kommentarzeilen fangen mit einer Raute (#) an.)
* @author Benni
* 
*/

public abstract class ConfigManager {
    
    //Standard Pfad zur KonfigDatei
    public static final String STANDARD_CONFIG_PATH = "project.conf";
    
    // Dateiobjekt das zur Konigurationsdatei zeigt
    private static File _file;
    
    //Map die die Einsttellungen enthlt
    private static Map<String, String> _configurationMap = new HashMap<String, String>();
    
    
    /**
     * Liet eine Konfiguration aus der StandardDatei
     * @return
     */
    public static Map<String, String> readConfigFromStandardFile() {
        return readConfigFromFile(new File(STANDARD_CONFIG_PATH));
    }
    
    
    /**
     * Liet eine Konfiguration aus einer angegebenen Datei
     * @param pFile
     * @return
     */
    public static Map<String, String> readConfigFromFile(File pFile) {
        // Variablen zum Einlesen
        String tempLine;
        BufferedReader br;
        
        _file = pFile;
        
        if(!_file.exists()) {
            writeFile();
        }
        
        try {
            br = new BufferedReader(new FileReader(pFile));
            // Diese Map soll alle Properties enthalten.
            Map<String, String> configurations = new HashMap<String, String>();
            
            while ((tempLine = br.readLine()) != null) {
                /*
                 * Rufe fr jede Zeile(templine) eine Methode auf, die diese Zeile in eine Map verwandelt und fge diese der Gesamt-Property-Map hinzu
                 */
                if (readConfigLine(tempLine) != null) {
                    configurations.putAll(readConfigLine(tempLine));
                }
                
            }
            
            _configurationMap = configurations;
            return configurations;
        }
        catch (Exception e) {
            Logger.log(LogLevel.ERROR, "Konnte keine Konfiguration lesen von: " + _file);
            e.printStackTrace();
            return null;
        }
        
    }
    
    
   /**
    * Liet eine einzelne Konfigurationszeile
    * @param line
    * @return
    */
   private static Map<String, String> readConfigLine(String line) {
        Map<String, String> configuration = new HashMap<String, String>();
        LinkedList<String> lineArguments = cutStringLine(line);
        
        if ((lineArguments.isEmpty()) || lineArguments.getFirst().startsWith("#")) {
            Logger.log(LogLevel.DEBUG, "Ignoriere: " + line);
            return null;
        }
        else if (lineArguments.size() == 2) {
            configuration.put(lineArguments.get(0), lineArguments.get(1));
            return configuration;
        }
        else {
            if(!line.trim().equals("")) {
                Logger.log(LogLevel.INFO, "Konnte Konfiguration nicht verarbeiten: " + line);
            }
            return null;
        }
    }
   
   
   /**
    * Diese Methode fgt eine Konfigurationszeile ein.
    * @param pConfigName
    * @param pConfigValue
    * @return
    */
   public static boolean addConfigValue(String pConfigName, String pConfigValue) {
       String tempLine;
       BufferedReader br;
   
       //Die Datei wurde noch garnicht gelesen oder erstellt
       if(_file == null || !_file.exists()) {
           return false;
       }
       //Den Wert gibts schon
       if(_configurationMap.containsKey(pConfigName)) {
           return false;
       }
       
       //Alles klar, fge den Wert in der ConfigRegistry hinzu und schreibe die Datei neu
       _configurationMap.put(pConfigName, pConfigValue);
       return writeFile();
   }
   
   
   /**
    * Diese Methode editiert den Wert des angegebenen Config-Namens.
    * @param pConfigName
    * @param pConfigValue
    * @return
    */
   public static boolean editConfigValue(String pConfigName, String pConfigValue) {
       String tempLine;
       BufferedReader br;
   
       //Die Datei wurde noch garnicht gelesen oder erstellt
       if(_file == null || !_file.exists()) {
           return false;
       }
       //Den Wert gibts nicht
       if(!_configurationMap.containsKey(pConfigName)) {
           return false;
       }
       
       //Alles klar, ndere den Wert in der ConfigRegistry und schreibe die Datei neu
       _configurationMap.remove(pConfigName);
       _configurationMap.put(pConfigName, pConfigValue);
       return writeFile();
   }
    


    /**
     * Diese Methode schreibt eine Zeile in das festgelegte Config-File
     * 
     * @param pKey
     *            Ein Name des Schlssel
     * @param pValue
     *            Ein Wert, der fr diesen Schlssel gespeichert werden soll.
     */
    private static boolean writeConfigLine(String pKey, String pValue) {
        if (pKey == null || pKey.equals("")) {
            Logger.log(LogLevel.INFO, "writeConfigLine: Key ist leer");
            return false;
        }
        if (pValue == null || pValue.equals("")) {
            Logger.log(LogLevel.INFO, "writeConfigLine: Value ist leer");
            return false;
        }
        
        // Schreibe Datei
        PrintWriter pw;
        
        try {
            //Schreibe nur wenn Config in der Registry und die Datei vorhanden ist
            if(getValueFromKey(pKey) != null && _file.exists()) {
                pw = new PrintWriter(new FileWriter(_file,true));
                pw.println(pKey + " = " + pValue+"\n");
                pw.close();
                return true;
            }
            else {
                return false;
            }
           
            
        }
        catch (Exception e) {
            Logger.log(LogLevel.INFO, "Fehler beim Schreiben eines Konfigeintrags: "+pValue + " in "+ _file);
            e.printStackTrace();
            return false;
        }
        
    }
    
    
    /**
     * Diese Methode gibt zu einem Schlssel einen Wert.
     * Der Schlssel muss eindeutig sein, die Liste muss bereits aktualisiert sein.
     * @param pKey
     * @return
     */
    public static String getValueFromKey(String pKey) {
        if(_configurationMap.size() == 0) {
            Logger.log(LogLevel.INFO, "getValueFromKey: Config Liste Leer, versuche Standarddatei zu lesen");
            readConfigFromStandardFile();
        }
        return (String) _configurationMap.get(pKey);
    }
    


    /**
     * Diese Methode schreibt ein Default-Konfig-File, wenn in der Registry aber bereits Konfigurationen festgelegt sind, 
     * dann werden diese auch mit angefgt.
     */
    private static boolean writeFile() {
        // Liste der ConfigLines
        LinkedList<String> configLines = new LinkedList<String>();
        
        // Basis-Header
        configLines
                .add("#############################################################################");
        configLines
                .add("#Diese Datei beinhaltet verschiedene Konfigurationsmglichkeiten            #");
        configLines
                .add("#Eine Einstellung muss dem Schema: NameDerEinstellung = Wert entsprechen    #");
        configLines
                .add("#Jede Abweichung von dieser Konvention wird ignoriert                       #");
        configLines
                .add("#Eine Raute (#) am Zeilenanfang wird ignoriert                              #");
        configLines
                .add("#############################################################################");
        configLines.add("");
        
        // Schreibe Datei
        PrintWriter pw;
        
        try {
            
            pw = new PrintWriter(new FileWriter(_file));
            // Schreibe jede Zeile aus der Liste
            for (int i = 0; i < configLines.size(); i++) {
                pw.println(configLines.get(i));
            }
            
            pw.close();
            //Schreibe, wenn vorhanden aus der Registry die Zeilen in die Datei, damit ist Datei und Registry immer aktuell.
            if(_configurationMap.size() > 0) {
                for (Map.Entry<String, String> configEntry : _configurationMap.entrySet()) {
                    writeConfigLine(configEntry.getKey(), configEntry.getValue());
                }
            }
            //Alles hat geklappt
            return true;
        }
        
        catch (Exception e) {
            Logger.log(LogLevel.ERROR, "Fehler beim Schreiben einer Default-Config: " + _file);
            e.printStackTrace();
            return false;
        }
        
    }
    


    /**
     * Diese Methode zerteilt den bergebenen String anhand von den Leerstellen. Im Gegensatz zur Split-Methode werden alle Leerstellen(variabel) ignoriert.
     * 
     * @param line
     *            Die Line die zerteilt werden soll.
     * @return Alle gefundenen Teilstrings werden als Liste zurckgegeben.
     */
    private static LinkedList<String> cutStringLine(String line) {
        // Hier sollen alle Teilstrings gespeichert werden.
        LinkedList<String> strings = new LinkedList<String>();
        
        /*
         * Suche die bergebene Line ab, speichere nur in Liste wenn der Teilstring selbst nicht leer ist
         */

        for (int i = 0; i < line.trim().split("=").length; i++) {
            strings.add(line.trim().split("=")[i].trim());
            
        }
        
        return strings;
    }
    
}
