package de.hwrBerlin.hwrchat.server.data;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import de.blackshark.debugger.Debugger;
import de.hwrBerlin.hwrchat.model.Group;
import de.hwrBerlin.hwrchat.model.User;
import de.hwrBerlin.hwrchat.model.messages.AMessage;
import de.hwrBerlin.hwrchat.server.persistence.DatabaseServerDAO;

/**
 * Diese Klasse soll das komplette Datenmodell beherbergen, dass fr den laufenden Serverbetrieb ntig ist. Es werden also alle Daten und auch Daten relationen in dieser Klasse zusammengefasst.
 * Verschiedene Methoden bieten dann einen Zugriff auf diese Daten und knnen diese auch verndern. Zustzlich existiert eine Schnittstelle zur Datenbank um die Daten dann auch permanent zu speichern.
 * 
 * @author Benni
 * 
 */
public class ServerDataDAO {
    
    private DatabaseServerDAO database = new DatabaseServerDAO();
    
    private List<User> _knownUserList = new ArrayList<User>();
    
    private List<Group> _knownGroupList = new ArrayList<Group>();
    
    // Maps zum schnellen Zugriff (nur zur Hilfe der Zuordnungen)
    private Map<Integer, User> _simpleIdtoUserMap = new HashMap<Integer, User>();
    
    private Map<Integer, Group> _simpleIdtoGroupMap = new HashMap<Integer, Group>();
    

    
    

    public ServerDataDAO() {
        // Flle zunchst alle Daten mit denen vom Server
        // Alle User (mit Relation der User die ein User kennt)
        _knownUserList = getAllUsers();
        
        // Alle Gruppen (mit Relation der User die in einer Gruppe sind)
        _knownGroupList = getAllGroups();
        
        //Erst jetzt knnen die Benutzer alle Gruppen Objekte zugeordnet bekommen
        updateUserKnowsGroupsRelations(_knownUserList, _knownGroupList);
        
    }
    


   



	// ///////////////////////////////USER///////////////////////////////////////////////////////////
    
    /**
     * Diese Methode findet alle Benutzer Es werden alle Benutzer der kennt Beziehung mit bercksichtigt.
     * 
     * @return
     */
    private List<User> getAllUsers() {
        // Alle Benutzer aus der Datenbank ohne die kennt Beziehung
        List<User> allDatabaseUsers = database.getAllUsers();
        
        // Fhre ein update der Maps aus
        updateSimpleIdToUserMap(allDatabaseUsers);
        
        /*
         * Jetzt muss noch fr jeden Benutzer aus der Liste ermittelt werden, wer wen kennt, kennt ein Benutzer einen anderen Benutzer wird die User Bean in die Liste der bekannten benutzer
         * eingeordnet, auerdem setze alle Gruppen die der Benutzer kennt!
         */
        for (User oneUser : allDatabaseUsers) {
            // Bekomme aus der Datenbank eine Liste von Usern die die aktuellen Benutzer kennen
            List<User> dummyUserList = database.getAllUserRelationsForOneUser("" + oneUser.getId());
            // Ordne jetzt diese jetzt dem aktellen oneUser Zu
            for (User dummyUser : dummyUserList) {
                // Und zwar aus der Map
                oneUser.addKnowingUsers(_simpleIdtoUserMap.get(dummyUser.getId()));
            }
        }
        

        // Gebe gefllte Liste zurck
        return allDatabaseUsers;
    }
    


    /**
     * Diese Methode kann eine gefllt User-Bean in die Datenplattform bernehmen. Die UserId wird von der Datenbank vergeben
     * 
     * @param user
     *            Eine (gefllte) User-Bean mit allen Inforamtionen ber den User
     */
    public int insertUser(User pUser) {
        // Einmal in die Datenbank aufnehmen (zuerst, da id bekannt sein muss)
        int userId = database.insertUser(pUser);
        
        // Und in unsere lokale Liste hinzufgen, aber vorher die id anfgen
        pUser.setId(userId);
        
        //Setze das verschlsselte Password        
        int test = pUser.getPassword();
        pUser.setPassword(test);
        
        
        addKnownUserList(pUser);  
        
        return userId;
    }
    


    /**
     * Diese Methode ndert die Daten eines Users. Es ist wichtig, dass die User-Bean so gefllt ist, wie die Informationen nachher in der Datenerhaltung stehen sollen. Es sollen also nicht nur
     * genderte Werte in dieser Bean stehen, sondern wirklich ALLE Diese Methode kann die attens Beziehung der Datenbank und auch nicht der lokalen Daten benutzen
     * 
     * @param user
     *            Eine (gefllte) User-Bean mit allen Informationen ber den User
     */
    public void updateUser(User user) {
        // suche die User TAbelle ab
        for (User possibleUser : _knownUserList) {
            // Finde User in der User Liste
            if (possibleUser.getId() == user.getId()) {
                // Update Informationen
                possibleUser.updateInformations(user);
            }
        }
        
        // mache dasselbe in der Datenbank
        database.updateUser(user);
        
    }
    


    /**
     * Diese Methode ruft einen User aus der Datenerhaltung ab. Bentigt wird dazu die UserId.
     * 
     * @param userId
     *            Zum finden eines Datensatzes muss eine userId bergeben werden.
     * @return User Es wird eine Bean mit den Informationen zurckgegeben(oder null wenn nicht in DB)
     */
    public User getUserById(int userId) {
        if (userId < 0) {
           Debugger.info(getClass(), "getUser: Keine UserId bekommen");
            return null;
        }
        
        // Durchsuche bekannte UserListe nach id
        for (User possibleUser : _knownUserList) {
            // Finde User in der User Liste
            if (possibleUser.getId() == userId) {
                // Gebe aktuellen User zurck
                return possibleUser;
            }
        }
        // Keinen User gefunden
        return null;
    }
    


    /**
     * Diese Methode ruft einen User aus der Datenerhaltung ab. Bentigt wird dazu der Username.
     * 
     * @param userName
     *            Zum finden eines Datensatzes muss ein UserName bergeben werden.
     * @return User Es wird eine Bean mit den Informationen zurckgegeben(oder null wenn nicht in DB)
     */
    public User getUserByName(String userName) {
        if (userName.trim().equals("") || userName == null) {
        	Debugger.info(getClass(), "getUser: Keinen Namen bekommen");
            return null;
        }
        
        // Durchsuche bekannte UserListe nach namen
        for (User possibleUser : _knownUserList) {
            // Finde User in der User Liste
            if (possibleUser.getName().trim().equals(userName.trim())) {
                // Gebe aktuellen User zurck
                return possibleUser;
            }
        }
        // Keinen User gefunden
        return null;
    }
    


    /**
     * Prft ob ein User in unserer Datenstruktur bekannt ist. Anhand der User Id
     * 
     * @param userId
     *            Der User der in der DB sein soll.
     * @return true, wenn er in der DB vorhanden ist, false wenn nicht
     */
    public boolean userIsInDataBaseById(int userId) {
        // Benutze Methode getUserById
        if (getUserById(userId) != null) {
            return true;
        }
        else {
            return false;
        }
    }
    


    /**
     * Prft ob ein User in unserer Datenstruktur bekannt ist. Anhand des USernames
     * 
     * @param userId
     *            Der User der in der DB sein soll.
     * @return true, wenn er in der DB vorhanden ist, false wenn nicht
     */
    public boolean userIsInDataBaseByName(String userName) {
        // Benutze Methode getUserByName
        if (getUserByName(userName) != null) {
            return true;
        }
        else {
            return false;
        }
    }
    


    /**
     * Lscht einen USer anhand der Id aus der Datenbank und den lokalen Daten
     * 
     * @param userId
     *            Eine UserId muss angegeben werden
     */
    public void deleteUserById(int userId) {
        // Durchsuche bekannte UserListe nach id
        for (User possibleUser : _knownUserList) {
            // Finde User in der User Liste
            if (possibleUser.getId() == userId) {
                // Lsche auch alle Beziehungen die er kennen knnte
                removeAllRelationsForUser(userId);
                // Lsche ihn aus der LIste
                _knownUserList.remove(possibleUser);
                break;
            }
        }
        
        // Lsche ihn auch aus der Datenbank
        database.deleteUser("" + userId);
    }
    


    /**
     * Diese Methode speichert die "kennt" Beziehung zwischen zwei Usern. Momentan ist diese Methode so gestaltet, dass nur in eine Richtung die Beziehung eingetragen wird. Wenn beide Personen sich
     * kennen muss also diese Methode zweimal mit vertauschten Parametern aufgrufen werden.
     * 
     * @param userIdOne
     * @param UserIdTwo
     */
    public void addUserKnowsUserRelation(int userIdOne, int userIdTwo) {
        // aktualisiere die Map mit der aktuellen UserList
        updateSimpleIdToUserMap(_knownUserList);
        
        // Aktualisiere die kennt Beziehung der UserBean fr userIdOne
        // Adde die User Bean fr UserIdTwo(vorher suche sie aus der Map)
        _simpleIdtoUserMap.get(userIdOne).getKnowingUsers().add(_simpleIdtoUserMap.get(userIdTwo));
        
        // Mache dasselbe auch in der Datenbank
        database.addUserKnowsUserRelation("" + userIdOne, "" + userIdTwo);
        
    }
    


    /**
     * Diese Methode gibt anhan der bergebenen UserId alle Benutzer zurck die der Benutzer mit dieser Id kennt.
     * 
     * @param userId
     *            Id von dessen Benutzer die bekannten Nutzer herausgesucht werden sollen.
     * @return Eine User Liste, wenn nutzer gefunden, null wenn user nicht gefunden wurde
     */
    public List<User> getAllUserRelationsForOneUser(int userId) {
        // Durchsuche bekannte UserListe nach id
        for (User possibleUser : _knownUserList) {
            // Finde User in der User Liste
            if (possibleUser.getId() == userId) {
                // Lsche ihn aus der LIste
                return possibleUser.getKnowingUsers();
            }
        }
        // Es wurde kein nutzer mit dieser Id gefunden
        return null;
        
    }
    


    /**
     * Diese Methode lscht eine Beziehung zwischen zwei Benutzern Die Beziehung wird beidseitig gelscht
     * 
     * @param userIdOne
     * @param userIdTwo
     */
    public void removeUserKnowsUserRelation(int userIdOne, int userIdTwo) {
        if (userIdOne < 0 || userIdTwo < 0) {
        	Debugger.info(getClass(), "removeUserKnowsUserRelation: UserId darf nicht kleiner -1 sein");
            return;
        }
        // Lokale nderung
        // Aktualisiere die IdtoUserMap
        updateSimpleIdToUserMap(_knownUserList);
        
        // Lsche in eine Richtung
        _simpleIdtoUserMap.get(userIdOne).getKnowingUsers().remove(_simpleIdtoUserMap.get(userIdTwo));
        
        // Lsche in die andere Richtung
        _simpleIdtoUserMap.get(userIdTwo).getKnowingUsers().remove(_simpleIdtoUserMap.get(userIdOne));
        
        // nderung in der Datenbank
        database.removeUserKnowsUserRelation("" + userIdOne, "" + userIdTwo);
        
    }
    


    /**
     * Diese Methode lscht alle Beziehungen die ein Benutzer hat. Die Beziehung wird beidseitig gelscht. Der User taucht also in keiner Relation mehr auf.
     * 
     * @param id
     */
    public void removeAllRelationsForUser(int userId) {
        if (userId < 0) {
        	Debugger.info(getClass(), "removeAllRelationsForUser: UserId darf nicht kleiner -1 sein");
            return;
        }
        
        // Lokale nderung
        // Aktualisere Map
        updateSimpleIdToUserMap(_knownUserList);
        User userToRemove = _simpleIdtoUserMap.get(userId);
        // Es muss fr jeden Nutzer die Liste der knowing Users aktualisiert werden
        for (User knowingUser : _knownUserList) {
            
            knowingUser.getKnowingUsers().remove(userToRemove);
            
        }
        
        // Lsche nun auch in der Datenbank
        database.removeAllRelationsForUser("" + userId);
        
    }
    
    /**
     * Diese Methode sucht anhand eines Usernamens,
     * User die den angegeben Namen beinhalten.
     * 
     * @param userNickName
     */
    public List<User> searchForUsers(String userNickName) {
        if (userNickName.trim().equals("") || userNickName == null) {
        	Debugger.info(getClass(), "searchForUsers: User name darf nicht leer sein");
            return null;
        }
        
        List<User> foundUserList = new ArrayList<User>();
        //Suche jeden User der Liste durch
        for (User user : _knownUserList) {
			if(user.getName().contains(userNickName)) {
			foundUserList.add(user);	
			}
		}
        
        return foundUserList;
        
    }
    
    /**
     * Prft ob die beiden NutzerIds sich kennen.
     * Es wird davon ausgegangen, dass bereits beide Bezihungen bestehen, es wird also nur eine Richtng
     * geprft!
     * @param userIdOne
     * @param userIdTwo
     * @return
     */
    public boolean userKnowsUser(int userIdOne, int userIdTwo) {
    	boolean theyKnow = false;
    	//Ist denn schon eine Beziehung vorhanden?
		for (User userfriend : getAllUserRelationsForOneUser(userIdOne)) {
			if(userfriend.getId() == userIdTwo) {
				theyKnow = true;
			}
		}
		
		return theyKnow;
    	
    }
    
    // ////////////////GRUPPEN///////////////////////////////
    
    /**
     * Diese Methode holt alle Gruppen aus der Datenbank und verknpft diese gleich mit den echten User-Objekten Es sollte also zuerst die User-Liste aufgebaut sein.
     * 
     * @return
     */
    private List<Group> getAllGroups() {
        // Alle Gruppen aus der Datenbank ohne die kennt Beziehung und nur Dummy Werten fr das Founder Attribut
        List<Group> allDatabaseGroups = database.getAllGroups();
        
        // Fhre ein update der Map aus
        updateSimpleIdToGroupMap(allDatabaseGroups);
        
        // Jetzt muss fr jeden Founder der Gruppe eine echte UserBean zugeordnet werden
        // Auerdem mssen die User ermittelt werden, die dieser Gruppe angehrig sind
        // Update also zuerst die USer Map
        updateSimpleIdToUserMap(_knownUserList);
        
        for (Group group : allDatabaseGroups) {
            int founderUserId = group.getFounder().getId();
            // Setze den Founder als das Objekt der Userliste mit der Id
            group.setFounder(_simpleIdtoUserMap.get(founderUserId));
            
            // Ermittel nun die Liste der User die der Gruppe angehren
            List<User> dummyUserAttendsGroup = database.getAllUsersInGroup(group.getId());
            
            // Die Liste dieser Dummy User muss nun noch in die lokale Liste der Liste umgewandelt werden.
            for (User dummyUserInGroup : dummyUserAttendsGroup) {
                User realLocalUser = _simpleIdtoUserMap.get(dummyUserInGroup.getId());
                group.getUser().add(realLocalUser);
            }
            
        }
        
        return allDatabaseGroups;
    }
    


    /**
     * Diese Methode kann eine gefllt Group-Bean in die Datenstruktur schreiben.
     * 
     * @param group
     *            Eine (gefllte) Group-Bean mit allen Inforamtionen ber die Gruppe
     */
    public int insertGroup(Group group) {
        // Trage zunchst in die Datenbankein um id zu erhalten
        int dataBaseGroupId = database.insertGroup(group);
        
        // Reihe jetzt in die Liste der lokalen Gruppen ein, trage vorher aber id ein
        group.setId(dataBaseGroupId);
        _knownGroupList.add(group);
        return dataBaseGroupId;
    }
    


    /**
     * Diese Methode kann eine gefllt Group-Bean aus den Datenstrukturen herausholen.
     * 
     * @param groupId
     *            es Muss eine GroupId angegben werden.
     */
    public Group getGroupById(int groupId) {
        // Gehe alle Gruppen durch
        for (Group possibleGroup : _knownGroupList) {
            // Suche anhand der id, gebe zurck fals gefunden
            if (possibleGroup.getId() == groupId) {
                return possibleGroup;
            }
        }
        // Es wurde keine Gruppe mit dieser Id gefunden
        return null;
        
    }
    


    /**
     * Diese Methode kann eine gefllt Group-Bean aus den Datenstrukturen herausholen.
     * 
     * @param groupName
     *            es Muss ein Gruppenname angegben werden.
     */
    public Group getGroupByName(String groupName) {
        // Gehe alle Gruppen durch
        for (Group possibleGroup : _knownGroupList) {
            // Suche anhand des Namens, gebe zurck fals gefunden
            if (possibleGroup.getName().trim().equals(groupName.trim())) {
                return possibleGroup;
            }
        }
        // Es wurde keine Gruppe mit diesem Namen gefunden
        return null;
        
    }
    


    /**
     * Diese Methode kann eine Beziehung zwischen einem User und einer Gruppe eintragen.
     * 
     * @param userId
     * @param groupId
     */
    public void attendGroup(int userId, int groupId) {
        // Trage die Beziehung erst lokal ein
        updateSimpleIdToGroupMap(_knownGroupList);
        updateSimpleIdToUserMap(_knownUserList);
        
        // Ordne der Gruppe, gefunden durch eine Map einen User zu, der auch durch eine Map gefunden wurde
        _simpleIdtoGroupMap.get(groupId).getUser().add(_simpleIdtoUserMap.get(userId));
        
        //Ordne dem User auch die Gruppe zu
        _simpleIdtoUserMap.get(userId).addKnowingGroup(_simpleIdtoGroupMap.get(groupId));
        
        // Mache dasselbe auf Datenbankebene
        database.attendGroup("" + userId, "" + groupId);
    }
    


    /**
     * Diese Methode gibt alle Benutzer zurck, die in einer Gruppe vorhanden sind. durchsucht
     * 
     * @param groupId
     * @return
     */
    public List<User> getAllUsersInGroup(int groupId) {
        
        // Durchsuche die komplette Gruppenlist
        for (Group knownGroup : _knownGroupList) {
            if (knownGroup.getId() == groupId) {
                return knownGroup.getUser();
            }
        }
        
        // Es wurde keine Gruppe mit dieser id gefundne
        return null;
    }
    


    /**
     * Diese Methode lscht eine Gruppe aus der Datenbank, und auch aus den lokalen Datenstrukturen, es werden auch die Realtionen mit bedacht
     * 
     * @param groupId
     */
    public void removeGroup(int groupId) {
        // Lsche lokal, durchsuche bis zur Gruppe mit dieser Id
        for (Group knownGroup : _knownGroupList) {
            if (knownGroup.getId() == groupId) {
                _knownGroupList.remove(knownGroup);
            }
        }
        
        // Lsche in der Datenbank(Hier wird die attends Beziehung auch gelscht
        database.removeGroup("" + groupId);
    }
    


    /**
     * Diese Methode lscht einen User aus allen Gruppen relationen Relationen fr eine UserId
     * 
     * @param userId
     */
    public void removeUserFromAttendsRelation(int userId) {
        // Auf lokaler Ebene
        // Hole User Objekt das gelscht werden soll
        updateSimpleIdToUserMap(_knownUserList);
        User deleteUserFromAllGroups = _simpleIdtoUserMap.get(userId);
        // Gehe alle Gruppen durch
        for (Group knownGroup : _knownGroupList) {
            knownGroup.getUser().remove(deleteUserFromAllGroups);
        }
        
        // Mache dasselbe auch auf Datenbank ebene
        database.removeUserFromAttendsRelation("" + userId);
        
    }
    


    // ////////////////Hilfsmethoden////////////////////////
    
    /**
     * Diese Methode aktualisiert die Map die eine id zu einem USer zuordnen kann
     * 
     * @param allDatabaseUsers
     */
    public void updateSimpleIdToUserMap(List<User> allDatabaseUsers) {
        // Lsche alle Daten
        _simpleIdtoUserMap.clear();
        
        // Ordne die UserLise einer Map zu um schnell von der id auf den User zu schlieen
        for (User user : allDatabaseUsers) {
            _simpleIdtoUserMap.put(user.getId(), user);
        }
    }
    
    /**
     * Diese Methode aktualisiert die Map die eine id zu einem USer zuordnen kann
     * Aber es wird die aktuelle _knownUserList
     * 
     * @param allDatabaseUsers
     */
    public void updateSimpleIdToUserMap() {
        // Lsche alle Daten
        _simpleIdtoUserMap.clear();
        
        // Ordne die UserLise einer Map zu um schnell von der id auf den User zu schlieen
        for (User user : _knownUserList) {
            _simpleIdtoUserMap.put(user.getId(), user);
        }
    }
    


    /**
     * Diese Methode aktualisiert die Map die eine id zu einer Gruppe zuordnen kann
     * 
     * @param allDatabaseUsers
     */
    public void updateSimpleIdToGroupMap(List<Group> allDatabaseGroups) {
        // Lsche alle Daten
        _simpleIdtoGroupMap.clear();
        
        // Ordne die UserLise einer Map zu um schnell von der id auf die Gruppe zu schlieen
        for (Group group : allDatabaseGroups) {
            _simpleIdtoGroupMap.put(group.getId(), group);
        }
    }
    
    
    /**
     * Diese Methode ordnet allen Usern die Gruppen Objekte zu, der sie angehren sollen.
     * @param _knownUserList2
     * @param _knownGroupList2
     */
    private void updateUserKnowsGroupsRelations(List<User> pKnownUsers,
			List<Group> pKnownGroups) {
    	
        for (User oneUser : pKnownUsers) {
            // Bekomme aus der Datenbank eine Liste von Gruppen an denen der Nutzer teilnimmt
            List<Group> dummyGroupList = database.getAllGroupsForUser(oneUser.getId());
            // Ordne jetzt diesen jetzt die echten Gruppen Objekte zu
            for (Group dummyGroup : dummyGroupList) {
                // Und zwar aus der Map
            	oneUser.addKnowingGroup(_simpleIdtoGroupMap.get(dummyGroup.getId()));
            }
        }
	}
    
    /**
     * Diese Methode sucht anhand eines Gruppennamens,
     * Gruppen die den angegeben Namen beinhalten.
     * 
     * @param groupName
     */
    public List<Group> searchForGroups(String groupName) {
        if (groupName.trim().equals("") || groupName == null) {
        	Debugger.info(getClass(), "searchForGroups: Group name darf nicht leer sein");
            return null;
        }
        
        List<Group> foundGroupList = new ArrayList<Group>();
        //Suche jeden Gruppe der Liste durch
        for (Group group : _knownGroupList) {
			if(group.getName().contains(groupName)) {
			    foundGroupList.add(group);	
			}
		}
        
        return foundGroupList;
        
    }
    
    
    /**
     * Prft ob die NutzerId schon in der Angegeben Gruppen Id
     * vorhanden ist
     * @param userId
     * @param groupId
     * @return
     */
    public boolean userIsInGroup(int userId, int groupId) {
        boolean theyKnow = false;
        //Ist denn bereits die Gruppe bekannt?
        User foundUser = getUserById(userId);
        for (Group userGroup : foundUser.getKnowingGroups()) {
            if(userGroup.getId() == groupId) {
                theyKnow = true;
            }
        }
        
        return theyKnow;
        
    }
    


    // ////////////////GETTER SETTER///////////////////////
    
    public List<User> getKnownUserList() {
        return _knownUserList;
    }
    


    public void setKnownUserList(List<User> pKnownUserList) {
        _knownUserList = pKnownUserList;
    }
    


    public void addKnownUserList(User pUser) {
        _knownUserList.add(pUser);
    }
    


    public List<Group> getKnownGroupList() {
        return _knownGroupList;
    }
    


    public void setKnownGroupList(List<Group> pKnownGroupList) {
        _knownGroupList = pKnownGroupList;
    }
    


    public void addKnownGroupList(Group pGroup) {
        _knownGroupList.add(pGroup);
    }
    
}
